(function() {
    var registerBlockType = wp.blocks.registerBlockType;
    var el = wp.element.createElement;
    var Fragment = wp.element.Fragment;
    var InspectorControls = wp.blockEditor ? wp.blockEditor.InspectorControls : wp.editor.InspectorControls;
    var PanelBody = wp.components.PanelBody;
    var TextControl = wp.components.TextControl;
    var ToggleControl = wp.components.ToggleControl;
    var SelectControl = wp.components.SelectControl;
    var RangeControl = wp.components.RangeControl;
    var ColorPicker = wp.components.ColorPicker;
    var RadioControl = wp.components.RadioControl;
    var Button = wp.components.Button;

    // Helper unit
    function ensureUnit(value, type) {
        if (!value) return '';
        if (value.match(/[a-z%]+$/)) return value;
        return value + (type === 'font' ? 'px' : 'em');
    }

    // URL generator
    function buildURL(attrs) {
        if (!attrs.ids) return 'about:blank';
        var url = 'https://koortickets.nl/ticketshop.php?embed&ids=' + encodeURIComponent(attrs.ids);

        // Colors
        for (var key in attrs.colors) {
            if (attrs.colors[key]) url += "&" + key + "color=" + encodeURIComponent(attrs.colors[key]);
        }

        // Visibility
        for (var key in attrs.visibility) {
            if (attrs.visibility[key]) url += "&" + key;
        }

        // FontSizes
        for (var key in attrs.fontSizes) {
            if (attrs.fontSizes[key]) {
                url += "&fontsizefor_" + key + "=" +
                    encodeURIComponent(ensureUnit(attrs.fontSizes[key], 'font'));
            }
        }

        // Spacing
        for (var key in attrs.spacing) {
            if (attrs.spacing[key]) {
                url += "&spacingfor_" + key + "=" +
                    encodeURIComponent(ensureUnit(attrs.spacing[key], 'spacing'));
            }
        }

        // Language
        if (attrs.language) url += "&" + attrs.language;
        // Input mode
        if (attrs.inputmode === 'radio') {
            url += "&useradio";
        }
        if (attrs.font) url += "&font=" + encodeURIComponent(attrs.font);

        return url;
    }

    const colorLabels = {
        accent: 'Accent color',
        background: 'Background color',
        font: 'Font color',
        table: 'Table color',
        button: 'Button color'
    };

    const fontSizeOrder = [
        "title", "subtitle", "artist",
        "website", "date", "time", "location",
        "product", "price", "stock",
        "promocode", "promocodeinput", "total", "button"
    ];

    const spacingOrder = [...fontSizeOrder];

    const visibilityLabels = {
        minimal: "Minimal",
        nowebsite: "Don't show website",
        nodate: "Don't show date",
        notime: "Don't show time",
        notitle: "Don't show title",
        noheader: "Don't show header",
        noartist: "Don't show artist",
        nolocation: "Don't show location",
        noflyer: "Don't show flyer",
        nosubtitle: "Don't show subtitle"
    };

    const defaultFontSizes = {
        website: '14px',
        product: '18px',
        title: '28px',
        subtitle: '18px',
        stock: '12px',
        price: '18px',
        promocode: '14px',
        promocodeinput: '14px',
        artist: '18px',
        location: '14px',
        date: '14px',
        time: '14px',
        total: '20px',
        button: '18px'
    };

    const defaultSpacing = {
        title: '0em',
        subtitle: '0em',
        artist: '0em',
        website: '0em',
        date: '0em',
        time: '0em',
        location: '0em',
        product: '0em',
        price: '0em',
        promocode: '0em',
        promocodeinput: '0em',
        total: '0em',
        button: '0em'
    };

    registerBlockType('koortickets/ticketshop', {
        title: 'KoorTickets Ticketshop',
icon: () => el('img', {
    src: 'https://koortickets.nl/general/koortickets_icon.svg',
    width: 24,
    height: 24,
    alt: ''
}),
        category: 'widgets',

        attributes: {
            ids: { type: 'string', default: '2' },
            font: { type: 'string', default: '' },
            inputmode: { type: 'string', default: 'dropdown' },
            colors: {
                type: 'object',
                default: { accent:'', background:'', font:'', table:'', button:'' }
            },
            visibility: {
                type: 'object',
                default: {
                    minimal:false,
                    noheader:false,
                    nowebsite:false,
                    nodate:false,
                    notime:false,
                    notitle:false,
                    noartist:false,
                    nolocation:false,
                    noflyer:false,
                    nosubtitle:false
                }
            },
            fontSizes: { type: 'object', default: { ...defaultFontSizes } },
            spacing:   { type: 'object', default: { ...defaultSpacing } },
            width: { type: 'string', default: '100%' },
            height: { type: 'string', default: '600px' }, // 👈 height attribute
            alignment: { type: 'string', default: 'center' },
            language: { type: 'string', default: 'nl' }
        },

        edit: function(props) {
            var attrs = props.attributes;
            var iframeURL = buildURL(attrs);

            return el(Fragment, {},

                el(InspectorControls, {},

                    // IDs
                    el(PanelBody, { title: 'General', initialOpen: true },
                        el('strong', { style: { display: 'block', marginBottom: '4px' } },
                            'Event IDs (;-separated)'
                        ),
                        el(TextControl, {
                            label: false,
                            value: attrs.ids,
                            onChange: v => props.setAttributes({ ids: v })
                        }),
                        el(RadioControl, {
                            label: 'Language',
                            selected: attrs.language,
                            options: [
                                { label: 'NL', value: 'nl' },
                                { label: 'EN', value: 'en' }
                            ],
                            onChange: v => props.setAttributes({ language: v })
                        }),
                        el(RadioControl, {
                            label: 'Input type',
                            selected: attrs.inputmode,
                            options: [
                                { label: 'Use dropdown', value: 'dropdown' },
                                { label: 'Use radio buttons', value: 'radio' }
                            ],
                            onChange: v => props.setAttributes({ inputmode: v })
                        }),
                        el(SelectControl, {
                            label: 'Font',
                            value: attrs.font,
                            options: [
                                { label: 'Default', value: '' },
                                { label: 'Julius Sans One', value: 'juliussansone' },
                                { label: 'Andika', value: 'andika' },
                                { label: 'Calibri', value: 'calibri' },
                                { label: 'Dancing Script', value: 'dancingscript' },
                                { label: 'Helvetica', value: 'helvetica' },
                                { label: 'Aladin', value: 'aladin' },
                                { label: 'Pacifico', value: 'pacifico' },
                                { label: 'Arvo', value: 'arvo' },
                                { label: 'PT Sans', value: 'ptsans' },
                                { label: 'Arial', value: 'arial' },
                                { label: 'Garamond', value: 'garamond' },
                                { label: 'Tahoma', value: 'tahoma' },
                                { label: 'Trebuchet MS', value: 'trebuchet' },
                                { label: 'Times New Roman', value: 'timesnewroman' },
                                { label: 'Georgia', value: 'georgia' },
                                { label: 'Verdana', value: 'verdana' },
                                { label: 'Courier New', value: 'couriernew' },
                                { label: 'Brush Script MT', value: 'brushscript' },
                                { label: 'Nunito', value: 'nunito' },
                                { label: 'Raleway', value: 'raleway' },
                                { label: 'TAN - PEARL', value: 'tanpearl' },
                                { label: 'Oswald', value: 'oswald' },
                                { label: 'Lato', value: 'lato' }
                            ],
                            onChange: v => props.setAttributes({ font: v })
                        })
                        
                    ),


                    // Colors
                    el(PanelBody, { title: 'Colors', initialOpen: false },
                        Object.keys(attrs.colors).map(key =>
                            el('div', { key, style: { marginBottom: '15px' } },
                                el('label', { style: { display:'block', marginBottom:'5px' } },
                                    colorLabels[key]
                                ),
                                el(ColorPicker, {
                                    color: attrs.colors[key] || '#ffffff',
                                    disableAlpha: true,
                                    onChangeComplete: v =>
                                        props.setAttributes({
                                            colors: { ...attrs.colors, [key]: v.hex.replace('#','') }
                                        })
                                }),
                                el(TextControl, {
                                    value: attrs.colors[key] || '',
                                    onChange: v =>
                                        props.setAttributes({
                                            colors: { ...attrs.colors, [key]: v.replace('#','') }
                                        })
                                })
                            )
                        )
                    ),

                    // Visibility
                    el(PanelBody, { title: 'Visibility', initialOpen: false },
                        Object.keys(attrs.visibility).map(key =>
                            el(ToggleControl, {
                                key,
                                label: visibilityLabels[key],
                                checked: attrs.visibility[key],
                                onChange: v =>
                                    props.setAttributes({
                                        visibility: { ...attrs.visibility, [key]: v }
                                    })
                            })
                        )
                    ),

                    // Font sizes
                    el(PanelBody, { title: 'Font Sizes', initialOpen: false },
                        el(Button, {
                            isSecondary: true,
                            style: { marginBottom: '10px' },
                            onClick: () =>
                                props.setAttributes({ fontSizes: { ...defaultFontSizes } })
                        }, 'Reset to defaults'),

                        fontSizeOrder.map(key =>
                            el(RangeControl, {
                                key,
                                label: key,
                                min: 2,
                                max: 72,
                                value: parseInt(attrs.fontSizes[key]) || parseInt(defaultFontSizes[key]),
                                onChange: v =>
                                    props.setAttributes({
                                        fontSizes: { ...attrs.fontSizes, [key]: v + 'px' }
                                    })
                            })
                        )
                    ),

                    // Spacing
                    el(PanelBody, { title: 'Font Spacings', initialOpen: false },
                        el(Button, {
                            isSecondary: true,
                            style: { marginBottom: '10px' },
                            onClick: () =>
                                props.setAttributes({ spacing: { ...defaultSpacing } })
                        }, 'Reset to defaults'),

                        spacingOrder.map(key =>
                            el(RangeControl, {
                                key,
                                label: key,
                                min: -0.15,
                                max: 0.15,
                                step: 0.01,
                                value: parseFloat(attrs.spacing[key]) || 0,
                                onChange: v =>
                                    props.setAttributes({
                                        spacing: { ...attrs.spacing, [key]: v + 'em' }
                                    })
                            })
                        )
                    ),

                    // ✅ NEW: Layout / Height
                    el(PanelBody, { title: 'Layout', initialOpen: false },
                        el(RangeControl, {
                            label: 'Height (px)',
                            min: 300,
                            max: 2000,
                            step: 10,
                            value: parseInt(attrs.height) || 600,
                            onChange: v =>
                                props.setAttributes({ height: v + 'px' })
                        })
                    )
                ),

                // Preview
                el('div', {
                    style:{
                        textAlign: attrs.alignment,
                        border: '1px solid #ccc',
                        padding: '10px'
                    }
                },
                    el('strong', {}, 'Preview:'),
                    el('iframe', {
                        src: iframeURL,
                        style:{
                            width: attrs.width,
                            height: attrs.height,
                            border: 0,
                            marginTop: '10px'
                        }
                    })
                )
            );
        },

        save: function() {
            return null;
        }
    });
})();
